# --
# Copyright (C) 2001-2021 OTRS AG, https://otrs.com/
# Copyright (C) 2021 Znuny GmbH, https://znuny.org/
# --
# This software comes with ABSOLUTELY NO WARRANTY. For details, see
# the enclosed file COPYING for license information (GPL). If you
# did not receive this file, see https://www.gnu.org/licenses/gpl-3.0.txt.
# --

package Kernel::System::Console::Command::List;

use strict;
use warnings;
use List::Util qw(max);

use Kernel::System::Console::InterfaceConsole;

use parent qw(Kernel::System::Console::BaseCommand);

our @ObjectDependencies = (
    'Kernel::Config',
    'Kernel::System::Main',
);

sub Configure {
    my ( $Self, %Param ) = @_;

    $Self->Description('List available commands.');

    return;
}

sub Run {
    my ( $Self, %Param ) = @_;

    my $ProductName    = $Kernel::OM->Get('Kernel::Config')->Get('ProductName');
    my $ProductVersion = $Kernel::OM->Get('Kernel::Config')->Get('Version');

    my $UsageText = "<green>$ProductName</green> (<yellow>$ProductVersion</yellow>)\n\n";
    $UsageText .= "<yellow>Usage:</yellow>\n";
    $UsageText .= " znuny.Console.pl command [options] [arguments]\n";
    $UsageText .= "\n<yellow>Options:</yellow>\n";

    OPTION:
    for my $Option ( @{ $Self->{_GlobalOptions} // [] } ) {
        next OPTION if $Option->{Invisible};
        my $OptionShort = "[--$Option->{Name}]";
        $UsageText .= sprintf " <green>%-40s</green> - %s", $OptionShort, $Option->{Description} . "\n";
    }
    $UsageText .= "\n<yellow>Available commands:</yellow>\n";

    my $PreviousCommandNameSpace = '';

    my @Commands         = $Self->ListAllCommands();
    my $MaxCommandLength = max map {length} @Commands;
    $MaxCommandLength -= length('Kernel::System::Console::Command::');

    COMMAND:
    for my $Command (@Commands) {
        my $CommandObject = $Kernel::OM->Get($Command);
        my $CommandName   = $CommandObject->Name();

        # Group by top-level namespace
        my ($CommandNamespace) = $CommandName =~ m/^([^:]+)::/smx;
        $CommandNamespace //= '';
        if ( $CommandNamespace ne $PreviousCommandNameSpace ) {
            $UsageText .= "<yellow>$CommandNamespace</yellow>\n";
            $PreviousCommandNameSpace = $CommandNamespace;
        }
        $UsageText .= sprintf(
            " <green>%-" . $MaxCommandLength . "s</green> - %s\n",
            $CommandName, $CommandObject->Description()
        );
    }

    $Self->Print($UsageText);

    return $Self->ExitCodeOk();
}

=head2 ListAllCommands()

Returns all available commands, sorted first by directory and then by file name.

    my @Commands = $CommandObject->ListAllCommands();

Returns:

    my @Commands = (
        'Kernel::System::Console::Command::Help',
        'Kernel::System::Console::Command::List',
        # ...
    );

=cut

sub ListAllCommands {
    my ( $Self, %Param ) = @_;

    my $ConfigObject = $Kernel::OM->Get('Kernel::Config');
    my $MainObject   = $Kernel::OM->Get('Kernel::System::Main');
    my $Home         = $ConfigObject->Get('Home');

    # Read commands from standard directory
    my @CommandFiles = $MainObject->DirectoryRead(
        Directory => $Home . '/Kernel/System/Console/Command',
        Filter    => '*.pm',
        Recursive => 1,
    );

    # Read commands from custom directory if it exists
    my $CustomCommandDir = $Home . '/Custom/Kernel/System/Console/Command';
    if ( -d $CustomCommandDir ) {
        my @CustomCommandFiles = $MainObject->DirectoryRead(
            Directory => $CustomCommandDir,
            Filter    => '*.pm',
            Recursive => 1,
        );
        push @CommandFiles, @CustomCommandFiles;
    }

    my @Commands;
    my %CommandSeen;

    COMMANDFILE:
    for my $CommandFile (@CommandFiles) {
        next COMMANDFILE if ( $CommandFile =~ m{/Internal/}xms );

        # Handle both standard and custom command files
        if ( $CommandFile =~ m{^.*Custom/(Kernel/System.*)[.]pm$}xmsg ) {

            # Custom command file
            $CommandFile = $1;
        }
        elsif ( $CommandFile =~ m{^.*(Kernel/System.*)[.]pm$}xmsg ) {

            # Standard command file
            $CommandFile = $1;
        }
        else {
            next COMMANDFILE;
        }

        $CommandFile =~ s{/+}{::}xmsg;

        # Skip if command already seen (avoid duplicates)
        next COMMANDFILE if $CommandSeen{$CommandFile};
        $CommandSeen{$CommandFile} = 1;

        push @Commands, $CommandFile;
    }

    # Sort first by directory, then by File
    my $Sort = sub {
        my ( $DirA, $FileA ) = split( /::(?=[^:]+$)/smx, $a );
        my ( $DirB, $FileB ) = split( /::(?=[^:]+$)/smx, $b );
        return $DirA cmp $DirB || $FileA cmp $FileB;
    };

    @Commands = sort $Sort @Commands;

    return @Commands;
}

1;
